/* 
This is a test sketch for the Autoshade 100105A Chopper Driver. It requires the driver mounted on an Arduino Uno with a
Adafruit RGBLCD Shield.

Speed change is made using the UP and DOWN buttons. + is CCW shaft rotation facing the motor. - is CW.
Voltage is updated when a speed change is made.
LEFT button starts the #1 (LEFT) motor at a constant speed. 
RIGHT button starts the #2 (RIGHT) motor at a constant speed. 
Display is not updated nor buttons read while motor is running to avoid any interrupt of motor drive I2C.
To stop motor, press RESET. 
At 2 amps, motor drivers overheat in about 2 minutes 
Max stepper speed is 750 steps per second due to I2C speed.

This program can be used with two motors but is intended to test the motor driver with two dummy loads 100112.
*/

#include <Wire.h>
#include <AccelStepper.h>
#include <Adafruit_MotorShield.h>

#include <Adafruit_RGBLCDShield.h>
#include <utility/Adafruit_MCP23017.h>

Adafruit_RGBLCDShield lcd = Adafruit_RGBLCDShield();
/******* Colors for the LCD backlight color ******************/
#define BLACK 0x0 // Use to save power
#define RED 0x1
#define YELLOW 0x3
#define GREEN 0x2
#define TEAL 0x6
#define BLUE 0x4
#define VIOLET 0x5
#define WHITE 0x7

#define STOP 0
#define RUN1 1
#define RUN2 2

/* The I2C address of the motor shield is set by bridging pads A0 & A1 on the board.
   A1=0 A0=0 are 0x7C. A1=0, A0=1 are 0x7D. A1=1, A0=0 are 0x7E. A1=1, A0=1 are 0x 7F.
   So up to 4 shields can be stacked.
   Create the motor shield object with the its I2C address:*/
Adafruit_MotorShield AFMS = Adafruit_MotorShield(0x7C); 
 

// Connect a stepper motor with 200 steps per revolution (1.8 degree)
// to motor port #1 (left) or motor port#2 (right). Change the '1' to '2' for port#2:
Adafruit_StepperMotor *myMotor1 = AFMS.getStepper(200, 1);
Adafruit_StepperMotor *myMotor2 = AFMS.getStepper(200, 2);

void forwardstep1() {myMotor1->onestep(FORWARD, DOUBLE);}
void backwardstep1() {myMotor1->onestep(BACKWARD, DOUBLE);}
AccelStepper stepper1(forwardstep1, backwardstep1); // use functions to step

void forwardstep2() {myMotor2->onestep(FORWARD, DOUBLE);}
void backwardstep2() {myMotor2->onestep(BACKWARD, DOUBLE);}
AccelStepper stepper2(forwardstep2, backwardstep2); // use functions to step

int stepsPerSec = 1; //This allows the torque to be tested at essentially zero speed.
int8_t mode = STOP;
int8_t buttons;

unsigned long currentms;
unsigned long lastms;

int   v12Pin = A0;
int   c12;

void setup() {
  c12 = analogRead(v12Pin); // volts = c12 * .01238 
  
/************ Setup LCD *****************/
  lcd.begin(16, 2); // set up LCD number of columns and rows 
  lcd.clear();
  lcd.setCursor(0,0); 
  lcd.setBacklight(WHITE);
  lcdPrint();
  
  
  AFMS.begin(); 
  Wire.setClock(400000L); // 400000L sets I2C clock for 400KHz (max for ATMEGA328P) to provide 797 steps/s vs 265 at default 100KHz. 
  //Note this statement must follow the AFMS.begin()
  
  stepper1.setMaxSpeed(1000.0f);
  stepper1.setSpeed(stepsPerSec); //+ or - steps per second
  myMotor1->release(); 
  
  stepper2.setMaxSpeed(1000.0f);
  stepper2.setSpeed(stepsPerSec); //+ or - steps per second
  myMotor2->release();  
  
  lastms = millis();
  }

void lcdPrint(void) //displays the selected speed (+ or -) and applied voltage.
{
  lcd.clear(); 
  lcd.setCursor(0,0);  
  lcd.print("Speed = "); 
  lcd.print(stepsPerSec); 
  lcd.setCursor(0,1); 
  if(mode!=STOP)lcd.print("Running");
  else {lcd.print("V12 = "); lcd.print(c12*.01238);} 
}

void loop() 
{     
  if(mode==RUN1) stepper1.runSpeed(); // runSpeed as often as possible. Do nothing else.
  else if(mode==RUN2) stepper2.runSpeed(); // runSpeed as often as possible. Do nothing else.
  else // in STOP mode set speed with buttons and start motor when ready.
  {
    currentms = millis(); 
    if((currentms - lastms) >= 100)// every 100 ms check if a button was pressed
    {
      lastms = currentms;
      c12 = analogRead(v12Pin); // volts = c12 * .01238 
      buttons = lcd.readButtons();  
      if (buttons ) 
      {
       delay(400);  
        switch(buttons) //Up & Down buttons change speed; right button starts motor2 running; left motor1.
        {
          case BUTTON_UP: stepsPerSec+=50; lcdPrint(); stepper1.setSpeed(stepsPerSec); stepper2.setSpeed(stepsPerSec); break;
          case BUTTON_DOWN: stepsPerSec-=50; lcdPrint(); stepper1.setSpeed(stepsPerSec); stepper2.setSpeed(stepsPerSec); break;
          case BUTTON_LEFT: mode=RUN1; lcdPrint(); break;
          case BUTTON_RIGHT: mode=RUN2; lcdPrint(); break;
          default: break;
        }
      }
    }     
  }   // end 100 ms process   
}  
 
  

